#!/bin/bash

# function: prompt
#
# parameters
#    $1 - prompt string
#    $2 - property value
#
# return
#    property value
#
# function will check to see if the property value is set.  If set, just use that value.
# If not set, prompt with the prompt string and allow the user to enter the value.

function prompt
{

    PROMPT_MSG=${1}
    PROMPT_VALUE=${2}

    variableEmpty "${PROMPT_VALUE}"
    if [ $? -eq 0 ]
    then
        if [ "${SHOW_PASSWORD}" -eq 0 ]
        then
            # turn off display echo
            stty -echo
        fi

        while [ "${PROMPT_VALUE}" = "" ]
        do
            echo
            echo -n "${PROMPT_MSG}: "
            read -r PROMPT_VALUE
            echo
        done

        if [ "${SHOW_PASSWORD}" -eq 0 ]
        then
            # turn on display echo
            stty echo
        fi
    fi
}

# function: copy
#
# parameters
#    $1 - source directory
#    $2 - source filename
#    $3 - target directory
#    $4 - (optional) target filename
#
# return
#    0 - successful
#    1 - unsuccessful
#
# function will copy the file from source directory to the target directory and maintain
# the source file name if target file name is not present

function copy
{
    SOURCE_DIR=${1}
    SOURCE_FILE=${2}
    TARGET_DIR=${3}
    TARGET_FILE=${4}

    # make sure that all the directory paths are created
    mkdir -p "${SOURCE_DIR}"
    mkdir -p "${TARGET_DIR}"

    if variableEmpty "${TARGET_FILE}"
    then
        cp "${SOURCE_DIR}"/"${SOURCE_FILE}" "${TARGET_DIR}"/"${SOURCE_FILE}"
    else
        cp "${SOURCE_DIR}"/"${SOURCE_FILE}" "${TARGET_DIR}"/"${TARGET_FILE}"
    fi
}

# function: isOracleDatabase
#
# parameters
#    none
#
# return
#    0 - if an oracle database
#    1 - if not an oracle database
#
# function will indicate if an oracle database is used

function isOracleDatabase
{
    TYPE_FROM_URL=$(echo "${DATABASE_URL}" | cut -f2 -d':')
    stringEqualsIgnoreCase "Oracle" "${TYPE_FROM_URL}"
    return $?
}

# function: isPostgresDatabase
#
# parameters
#    none
#
# return
#    0 - if an postgres database
#    1 - if not an postgres database
#
# function will indicate if a postgres database is used

function isPostgresDatabase
{
    TYPE_FROM_URL=$(echo "${DATABASE_URL}" | cut -f2 -d':')
    stringEqualsIgnoreCase "postgresql" "${TYPE_FROM_URL}"
    return $?
}

# function: calculateNumericValue
#
# parameters
#    $1 - version string in the format of 1.2.3
#
# return
#    numeric value representing the string format
#
# function will parse the format into 3 elements and assign a weight to each element value as follows:
#    example: 2.7.9
#        first element * 10000 + second element * 10 + third element
#        1 * 10000 + 2 * 10 + 3
# this should put the most emphasis on the first element and the resulting value will be returned for comparison

function calculateNumericalValue
{
    VERSION_NUMBER=${1}

    # assigned weights to take a format into a number, weights need to be more significant as you move through the format
    # the array is backwards so we can loop from right to left starting at index 0
    # 1.2.3 would use 3(weight of 1), 2(weight of 10) and 1(weight of 10000)
    WEIGHTS=(1 100 10000 100000 1000000)

    # determine the number of separators, the tr command will delete the periods and the -c take the complement
    # so 1.2.3 will end up with "..", the do a character count of the results
    NUM_OF_SEPARATORS=$(echo "${VERSION_NUMBER}" | tr -dc '.' | wc -c)

    # this is the field number that will be cut out of the version format and will vary based on the segments in the format
    # and will be adjusted based on the length of the format
    FIELD_NUMBER=1

    VALUE=0
    LOOP=${NUM_OF_SEPARATORS}
    while [ "${LOOP}" -ge 0 ]
    do
        TEMP_VALUE=$(cut -f${FIELD_NUMBER} -d'.' <<< "${VERSION_NUMBER}")
        FIELD_NUMBER=$((FIELD_NUMBER + 1))
        VALUE=$((VALUE + TEMP_VALUE * WEIGHTS[LOOP]))
        LOOP=$((LOOP - 1))
    done

    echo "${VALUE}"
}

# function: sameBaseDirectory
#
# parameters
#    $1 - base directory
#    $2 - directory to check against base directory
#
# return
#    0 - if the second directory has the same base directory
#    1 - if the second directory has a different base directory or if second directory isn't configured
#
# function will check to see if the second directory has the same base directory

function sameBaseDirectory
{
    BASE_DIR=${1}
    CHECK_DIR=${2}

    TEMP_DIR=${CHECK_DIR}

    SAME_BASE=1

    variableNotEmpty "${CHECK_DIR}"
    if [ $? -eq 0 ]
    then
        # loop on the second directory until the base matches or run out of directories
        while [ ${SAME_BASE} -eq 1 ] && [ "${TEMP_DIR}" != "/"  ]
        do
                if [ "${BASE_DIR}" = "${TEMP_DIR}" ]
                then
                        SAME_BASE=0
                else
                # this function will remove the last directory and return the remainder of the
                # directory path, right to left
                        TEMP_DIR=$(dirname "${TEMP_DIR}")
                fi
        done
    fi

    return ${SAME_BASE}
}

# function: stringLength
#
# parameters
#    $1 - string to determine the length of
#
# return
#    the length of the input string
#
# function will determine the length of the input string without the newline char

function stringLength
{
    TEMP_LENGTH=$(echo "${1}" | wc -c)
    # remove the count of the newline char
    LENGTH=$(expr "${TEMP_LENGTH}" - 1)

    echo "${LENGTH}"
}

# function: stringEqualsIgnoreCase
#
# parameters
#    $1 - first string value to compare
#    $2 - second string value to compare against
#
# return
#    0 - the two strings are equal, regardless of case
#    1 - the two strings are not equal, regardless of case
#
# function will convert the two input strings to upper case before comparing values

function stringEqualsIgnoreCase
{
        STR1=$(echo "${1}" | tr 'a-z' 'A-Z')
        STR2=$(echo "${2}" | tr 'a-z' 'A-Z')

        if [ "${STR1}" = "${STR2}" ]
        then
                return 0
        else
                return 1
        fi
}

# function: stringEquals
#
# parameters
#    $1 - first string value to compare
#    $2 - second string value to compare against
#
# return
#    0 - the two strings are equal, including case
#    1 - the two strings are not equal, including case
#
# function will compare the two input strings as passed in, case will matter

function stringEquals
{
        STR1=${1}
        STR2=${2}

        if [ "${STR1}" = "${STR2}" ]
        then
                return 0
        else
                return 1
        fi
}

# function: variableEmpty
#
# parameters
#    $1 - value to check for data
#
# return
#    0 - the input value is empty, no data assigned
#    1 - the input value is not empty, data is present
#
# function will check the input value to deterime if there is assigned data or not

function variableEmpty
{
    VAL=${1}

        if [ "${VAL}" = "" ]
        then
                return 0
        else
                return 1
        fi
}

# function: variableNotEmpty
#
# parameters
#    $1 - value to check for data
#
# return
#    0 - the input value is empty, no data assigned
#    1 - the input value is not empty, data is present
#
# function will check the input value to deterime if there is assigned data or not

function variableNotEmpty
{
    VAL=${1}

        if [ "${VAL}" = "" ]
        then
                return 1
        else
                return 0
        fi
}

# function: createDirectory
#
# parameters
#    $1 - directory path
#
# return
#    error if directory path can't be created
#
# function will check to see if directory is configured in cssinstaller_config.txt and create if needed

function createDirectory
{
    DIRECTORY=${1}

    variableNotEmpty "${DIRECTORY}"
    RC=$?
    if [ ${RC} -eq 0 ]
    then
        # -p option will create if needed and not fail if already exists
        mkdir -p "${DIRECTORY}" > tmp.$$ 2>&1
        RC=$?
        if [ ${RC} -ne 0 ]
        then
            TEMP_ERROR=$(cat tmp.$$)
            logPoint "Creating ${DIRECTORY} was unsuccessful.  Error: ${TEMP_ERROR}"

            generateFatalErrorMessage 103 "${DIRECTORY}" "${CONFIG_FILE}"
            rm -f tmp.$$ > /dev/null 2>&1
        fi

        rm -f tmp.$$ > /dev/null 2>&1
    fi
    return ${RC}
}

function setZipCSSVersion
{
    if [ -f "${INSTALLER_TEMP_SOFTWARE_DIRECTORY}"/"${APACHE_TOMCAT_CSS_WEBAPPS_CSS_WAR}" ]
    then

        unzip "${INSTALLER_TEMP_SOFTWARE_DIRECTORY}"/"${APACHE_TOMCAT_CSS_WEBAPPS_CSS_WAR}" "${METAINF_MANIFEST_MF}" > /dev/null 2>&1
        T_RC=$?
        if [ ${T_RC} -ne 0 ]
        then 
            generateFatalErrorMessage 133 "${INSTALLER_TEMP_SOFTWARE_DIRECTORY}"/"${APACHE_TOMCAT_CSS_WEBAPPS_CSS_WAR}" "$(pwd)"
        fi
        CSS_VERSION=$(cat "${METAINF_MANIFEST_MF}" 2>&1 | grep "Implementation-Version" | cut -f2 -d' '  | tr -d '\r')
        CSS_MAJOR_NO=$(cat "${METAINF_MANIFEST_MF}" 2>&1 | grep "Implementation-Version" | cut -f1 -d'.' | cut -f2 -d' ')
        logPoint "CSS ${CSS_VERSION} was found."
        rm -rf "${METAINF}"

        # Check if CSS_MAJOR_NO is a valid number, which is should be, but one can never be too careful.
        if ! [[ "$CSS_MAJOR_NO" =~ ^[0-9]+$ ]]; then
          generateFatalErrorMessage 154 "$CSS_MAJOR_NO"
        fi

        # Check if CSS_MAJOR_NO is 12 or greater. This is safe for 9.0 according to copilot, but will want to test it with installing 9+10.
        if [ "$CSS_MAJOR_NO" -ge 12 ]; then
          generateFatalErrorMessage 155 "$CSS_MAJOR_NO"
        fi
    fi
}

# function: setLatestVersion
#
# parameters
#    none
#
# sets LATEST_VERSION to the lastest version (4 digits)
# return
#    0 - have access to the internet
#    1 - don't have access to the internet
#
# function will get the latest version, and also let's us know if the Internet is available.

function setLatestVersion
{
    # test the the page exists to determine if we can get to the internet
    wget -O current_version.txt "${LATEST_VERSION_URL}" --timeout=12 --tries=1 > /dev/null 2>>"${LOG_FILE}"
    WGET_RC=$?
    if [ $WGET_RC  -eq 0 ] 
    then
        LATEST_VERSION=$(cat current_version.txt)
        return 0
    fi
    return 1

}

# function: initialize
#
# parameters
#    none
#
# return
#    none
#
# function will perform initialization steps whenever the script is executed, such as
# rolling logs and reading in the configuration file

function initialize
{
    # check and create necessary directories
    createDirectory "${LOG_DIRECTORY}" || exit 103
    createDirectory "${WORKING_DIRECTORY}"
    createDirectory "${ARTIFACT_REPOSITORY_DIRECTORY}" 
    createDirectory "${CSS_HOME}"
    createDirectory "${INSTALLER_DOWNLOADS_DIRECTORY}" 
    createDirectory "${INSTALLER_DOWNLOADS_SOFTWARE_DIRECTORY}" 
    createDirectory "${INSTALLER_DOWNLOADS_WALLETS_DIRECTORY}"
    createDirectory "${CSS_DATABASE_WALLET_DIRECTORY}"
    createDirectory "${INSTALLER_TEMP_CONFIG_DIRECTORY}" 
    createDirectory "${INSTALLER_TEMP_DRIVER_DIRECTORY}" 
    createDirectory "${INSTALLER_TEMP_SOFTWARE_DIRECTORY}" 
    createDirectory "${INSTALLER_TEMP_INFLUXDB_DIRECTORY}"
    createDirectory "${INFLUX_DB_DIRECTORY}"
    createDirectory "${INFLUX_DB_CONFIG_DIRECTORY}"
    createDirectory "${INFLUX_DB_VSM_DATA_DIRECTORY}"
    chmod 700 ${CSS_DATABASE_WALLET_DIRECTORY}
    createDirectory "${CSS_SSO_DIRECTORY}"
    chmod 700 ${CSS_SSO_DIRECTORY}

    # setup the logs to be new for the next execution of the script
    rollLogs

    if [ "${PREREQ_MODE}" -eq 0 ]
    then
        if [ "${EXECUTION_STATE}" -eq "${INSTALL}" ]
        then
            logPoint "Initializing the start of the installation"
        else
            logPoint "Initializing the start of the upgrade"
        fi
    else
            logPoint "Initializing the start of the prerequisite validation"
    fi

    # create the directory only if a PostgreSQL database as the oracle directory will be on the database server
    isPostgresDatabase
    if [ $? -eq 0 ]
    then
        createDirectory "${BACKUP_DATABASE_DIRECTORY}"
    fi

    setupMountPointArrays

}

function setupMountPointArrays
{
# initialize array with all mount points and available space
    INDEX=0
    for MNT_PNT in $(df -Pm | sort | grep -Ev "Available|tmpfs" | tr -s ' ' | cut -f1 -d' ')
    do
        MNT_PNTS[${INDEX}]=${MNT_PNT}
        AVAILABLE_SPACE[${INDEX}]=$(df -Pm | grep "${MNT_PNT} " | tr -s ' ' | cut -f4 -d' ')

        INDEX=$((INDEX + 1))
    done
}


# function: usage
#
# parameters
#    none
#
# return
#    none
#
# function will display the options that can be utilized with the command

function usage
{
    MESSAGE=${1}

    clear

    echo "${MESSAGE}"
    echo
    echo "${0} -c <absolute path to cssinstaller_config.txt> -v <latest or exact version> [-h -p -n -d -m -s -i -a -w]"
    echo "    c /<path to>/cssinstaller_config.txt - (required) points to the absolute path to the saved cssinstaller_config.txt file that contains the installation configurations, shouldn't stay in installer directory."
    echo "    v version - (required) indicates which version to install/upgrade. possible values are word latest or exact version like 6.5.0.1, 6.0.0.4. If latest is provided then most recent version of CSS will be used."
    echo "    h - (optional) displays usage."
    echo "    p - (optional) only checks the prerequisites to verify that the installation parameters are appropriate."
    echo "    d - (optional) indicates to run as a is a Docker installation. Provide flag for Docker install."
    echo "    m - (optional) changes the mode from standard mode to proof of value (pov) mode, which reduces prerequisite values."
    echo "    s - (optional) will show the password as it is typed in, otherwise just blank (no value if password is in config file."
    echo "    i - (optional) skip backing up the database. Only use this if you have backed up the database another way."
    echo "    a - (optional) skip backing up the artifacts folder. Only use this if you have backed up the artifacts another way."
    echo

    exit 1
}

# function: processArgs
#
# parameters
#    $@ - all passed in arguments from command line
#
# return
#    none
#
# function will parse the arguments from the command line and set appropriate flags based on the
# functionality needed by each argument

function processArgs
{
    # set default values for any argument flags.
    IGNORE_ARTIFACT_BACKUP=0
    PREREQ_MODE=0
    IGNORE_DB_BACKUP=0
    SHOW_PASSWORD=0
    MODE=${STANDARD}
    DOCKER_INSTALL=0

    while getopts "phdmc:v:siaw" ARG
    do
          case ${ARG} in
              p) 
                # turn on prereq mode when "p" is used as an argument, prereq mode will 
                # just run through the prerequisite checks and let user know what might be 
                # missing or they are ready to run the install
                PREREQ_MODE=1
                ;;
              m)
                # change the mode from standard to pov, which will reduce prerequisites limits
                MODE=1
                ;;
              c)
                # pass in the location of the installation cssinstaller_config.txt file, shouldn't be in installer directory
                # of it will get overlayed on an upgrade
                CONFIG_FILE=${OPTARG}
                ;;
              v)
                # pass in the version of CSS you want to install / upgrade to.
                DESIRED_VERSION=${OPTARG}
                ;;
              i) 
                # a way to skip the database backup
                IGNORE_DB_BACKUP=1
                ;;
              a) 
                # a way to skip the artifacts backup
                IGNORE_ARTIFACT_BACKUP=1
                ;;
              s) 
                # show the password as the user types it in
                SHOW_PASSWORD=1
                ;;
              d)
                # docker install flag
                DOCKER_INSTALL=1
                ;;
              *|h) 
                # display usage
                usage
                ;;
          esac

    done

    if [ "${CONFIG_FILE}" = "" ]
    then
        usage "Must use -c option to point to the installation cssinstaller_config.txt file, shouldn't be in installation directory due to possible overlay (ex. /dir/cssinstaller_config.txt"
    fi

    if [ ! -f "${CONFIG_FILE}" ]
    then
        usage "-c was specified, but ${CONFIG_FILE} doesn't exist. Make sure that you specify the file path like -c/<path to>/cssinstaller_config.txt"
    fi
    
    if [ "${DESIRED_VERSION}" = "" ]
    then
        usage "Must use -v option to specify the desired version of CSS to install, either latest or a version like 6.5.0.0"
    fi

    DESIRED_VERSION=$( tr '[:upper:]' '[:lower:]' <<<"$DESIRED_VERSION" )
    
    version_regex="[0-9]+\.[0-9]+\.[0-9]+\.[0-9]+"
    if [ "${DESIRED_VERSION}" = "latest" ]
    then
        echo "Latest CSS version specified."
    elif [[ "$DESIRED_VERSION" =~ $version_regex ]]
    then
        echo "CSS version ${DESIRED_VERSION} specified."
    else
        usage "-v needs a valid CSS version, either latest, or a 4 segment version number like 6.5.0.0"
    fi
    
}

# setupBackupVars uses CSS_VERSION.
function setupBackupVars
{
    if [ "${DOCKER_INSTALL}" -eq 1 ]
    then
        UPGRADE_BACKUP_DIRECTORY=${UPGRADE_BASE_DIRECTORY}/upgrade_backups/${CSS_VERSION}
    else
        UPGRADE_BACKUP_DIRECTORY=${UPGRADE_BASE_DIRECTORY}/upgrade_backups/pre_${CSS_VERSION}
    fi
    
    UPGRADE_BACKUP_ARTIFACTS_DIRECTORY=${UPGRADE_BACKUP_DIRECTORY}/artifacts
    UPGRADE_BACKUP_TOMCAT_DIRECTORY=${UPGRADE_BACKUP_DIRECTORY}/tomcat
    UPGRADE_BACKUP_PLUGINS_DIRECTORY=${UPGRADE_BACKUP_DIRECTORY}/plugins
    UPGRADE_BACKUP_CSS_TOMCAT_BIN_DIRECTORY=${UPGRADE_BACKUP_TOMCAT_DIRECTORY}/${APACHE_TOMCAT_CSS_BIN}

    createDirectory "${UPGRADE_BACKUP_ARTIFACTS_DIRECTORY}" 
    createDirectory "${UPGRADE_BACKUP_TOMCAT_DIRECTORY}" 
    createDirectory "${UPGRADE_BACKUP_PLUGINS_DIRECTORY}" 
}

# function: unzipper
#
# parameters
#    $1 - the input folder name
#    $2 - the output folder name
#    $3 - "ignore" will prevent fatal error message 133 from being thrown.
#
# return
#    the return code from unzip
#

function unzipper
{
    unzip "${1}"/*.zip -d "${2}" > /dev/null 2>>"${LOG_FILE}"
    ZIP_RC=$?
    if [ ${ZIP_RC} -eq 11 ]
    then
        NO_OF_ZIPS=$(ls "${1}"/*.zip | wc -l)
        generateFatalErrorMessage 107 "${1}" "${NO_OF_ZIPS}" "INSTALLER_DOWNLOADS_DIRECTORY"
    elif [ ${ZIP_RC} -ne 0 ] && [ "${3}" != "ignore" ]
    then
        generateFatalErrorMessage 133 "${1}/*.zip" "${2}"
    fi
    return ${ZIP_RC}
}


# function: requireGbFree
#
# parameters
#    $1 - The path to check for space in
#    $2 - Number, how many GB to require in that location 
#
# return
#    none. Throws fatal 146 and exits 146 if space is insufficient.
#
function requireGbFree
{
    MB_FREE=$(echo $(($2 * 1000))) #Round numbers will look cleaner.
    checkForSpace "installer" "${MB_FREE}" "${1}" "SKIP" || exitFatalErrorMessage 146 "${1}" $

}

# function: sourceSetEnvOverride
#
# parameters
#    none
#
# return
#    none
#
# function will check to see if the setenvoverride file exists and update config to match.
function sourceSetEnvOverride
{
    # check to see if the setenvoverride file even exists. It won't for docker or for old versions of FD.
    if [ -f "${CSS_HOME}/${APACHE_TOMCAT_CSS_BIN}/${SETENVOVERRIDE_FILE_NAME}" ]
    then
        logPoint "Sourcing in ${SETENVOVERRIDE_FILE_NAME} to use the artifact and working directories from your previous installation."
        HERE=$(pwd)
        cdOrFail "${CSS_HOME}/${APACHE_TOMCAT_CSS_BIN}"
        CATALINA_HOME="${CSS_HOME}/${APACHE_TOMCAT_CSS}"
        . "./${SETENVOVERRIDE_FILE_NAME}"
        cdOrFail "${HERE}"

        if [ "${ARTIFACT_REPOSITORY_DIRECTORY}" != "${FD_REPOSITORY_DIRECTORY}" ] 
        then
            logPoint "ARTIFACT_REPOSITORY_DIRECTORY updated to ${FD_REPOSITORY_DIRECTORY} using your setenvoverride.sh file."
            ARTIFACT_REPOSITORY_DIRECTORY="${FD_REPOSITORY_DIRECTORY}"
        fi

        if [ "${WORKING_DIRECTORY}" != "${FD_APPLICATION_DIRECTORY}" ] 
        then
            logPoint "WORKING_DIRECTORY updated to ${FD_APPLICATION_DIRECTORY} using your setenvoverride.sh file."
            WORKING_DIRECTORY="${FD_APPLICATION_DIRECTORY}"
            
            # adjust INFLUX_DB_VSM_DATA_DIRECTORY variable
            INFLUX_DB_VSM_DATA_DIRECTORY=${WORKING_DIRECTORY}/vsm
            createDirectory "${INFLUX_DB_VSM_DATA_DIRECTORY}"
        fi
    fi
    
    logPoint "ARTIFACT_REPOSITORY_DIRECTORY=${ARTIFACT_REPOSITORY_DIRECTORY}"
    logPoint "WORKING_DIRECTORY=${WORKING_DIRECTORY}"    
}

function makeFakeWar
{
    createDirectory "${CSS_HOME}/${APACHE_TOMCAT_CSS_WEBAPPS}"
    if [ ! -f "${CSS_HOME}/${APACHE_TOMCAT_CSS_WEBAPPS_CSS_WAR}" ]
    then
        echo a > "${CSS_HOME}/${APACHE_TOMCAT_CSS_WEBAPPS_CSS_WAR}"
    fi
}

function tomcatDirDeleteFailedDuringUpgrade
{
    #recreate a fake war in case something later fails before we have the new war.
    makeFakeWar
    exitFatalErrorMessage 143 "${CSS_HOME}"/"${APACHE_TOMCAT_CSS}"
}

# function: goOrStop
#
# parameters
#    none
#
# return
#    none
#
# function will check if the user wants to continue or stop.
function goOrStop
{
    GOSTOP=""
    # Remember what was the value of SHOW_PASSWORD first
    REMEMBER_SHOW_PASSWORD=${SHOW_PASSWORD}
    SHOW_PASSWORD=1
    while variableEmpty "${GOSTOP}"
    do
        prompt "Enter Y to continue or N to stop." "${GOSTOP}"
        GOSTOP=${PROMPT_VALUE}
        if [ "${GOSTOP}" = "N" ] || [ "${GOSTOP}" = "n" ] 
        then
            generateFatalErrorMessage 148 
            #Fatal doesn't stop during prereqs, so exit.
            exit 148
        elif [ "${GOSTOP}" = "Y" ] || [ "${GOSTOP}" = "y" ]
        then
            SHOW_PASSWORD=${REMEMBER_SHOW_PASSWORD}
            return 0
        else
            GOSTOP=""
        fi
    done
    
}

# function: cdOrFail
#
# parameters
#    $1 The directory to attempt to cd to.
#
# return
#    none
#
# function will cd to the directory of exit with a fatal, even during prereqs.
function cdOrFail
{
    cd "${1}" || exitFatalErrorMessage 151 "${1}"
}


function handleDbError
{
  if [ "${1}" -eq 14 ]
  	then
  		generateFatalErrorMessage 153
  	elif [ "${1}" -gt 11 ]
  	then
  		generateFatalErrorMessage 116 "${CONFIG_FILE}"
  	elif [ "${1}" -gt 10 ]
  	then
  	  generateFatalErrorMessage 115 "${CONFIG_FILE}" "${2}"
  	elif [ "${1}" -gt 0 ]
  	then
  		generateFatalErrorMessage 119 "${LOG_DIRECTORY}"
  	fi
}