#!/bin/bash 

# function: determineState
#
# parameters
#    none
#
# return
#    0 - state of execution is for an installation
#    1 - state of execution is for an upgrade
#
# function will determine if there is already an installation, thus an upgrade is needed else install

function determineState
{
    FLEXDEPLOY_STATE=${INSTALL}

    if [ "${IS_WEBLOGIC}" -eq 1 ]
    then
        logPoint "Running in upgrade from Weblogic to Tomcat mode."
        FLEXDEPLOY_STATE=${UPGRADE_FROM_WLS}
    elif [ "${DOCKER_INSTALL}" -eq 1 ]
    then
        determineDockerState
        FLEXDEPLOY_STATE=$?
        if [ "${FLEXDEPLOY_STATE}" -eq "${INSTALL}" ]
        then
            IGNORE_DB_BACKUP=1
        fi
    elif [ -f "${FLEXDEPLOY_HOME}/${APACHE_TOMCAT_FLEXDEPLOY_WEBAPPS_FLEXDEPLOY_WAR}" ] || [ -d "${FLEXDEPLOY_HOME}/${APACHE_TOMCAT_FLEXDEPLOY_WEBAPPS}/flexdeploy" ]
    then
        logPoint "Upgrading existing FlexDeploy installation found at ${FLEXDEPLOY_HOME}."
        FLEXDEPLOY_STATE=${UPGRADE}
    else 
        logPoint "Installing new FlexDeploy instance at ${FLEXDEPLOY_HOME} because ${FLEXDEPLOY_HOME}/${APACHE_TOMCAT_FLEXDEPLOY_WEBAPPS_FLEXDEPLOY_WAR} was not found."
        logPoint "If you wanted to upgrade, choose N and check the FLEXDEPLOY_HOME property in your config file."
        logPoint "If you are upgrading from Weblogic, choose N and run with the -w argument."
        logPoint "If the war has already been deleted, choose N and run the command {mkdir -p ${FLEXDEPLOY_HOME}/apache-tomcat-flexdeploy/webapps && echo > ${FLEXDEPLOY_HOME}/${APACHE_TOMCAT_FLEXDEPLOY_WEBAPPS_FLEXDEPLOY_WAR}}"
        IGNORE_DB_BACKUP=1
    fi    
    logPoint "Verify the installation mode above."
    if [ "${PREREQ_MODE}" -eq 1 ]
    then
        logPoint "Running in prereq mode. Nothing will be changed. The prompt is to assist in verification of the mode selected by the installer."
    fi
    goOrStop 
    return "${FLEXDEPLOY_STATE}"
}

# function: prepareSoftware
#
# parameters
#    none
#
# return
#    none
#
# function will determine if the FlexDeploy software needs to be downloaded or unpackaged and ready for installation

function prepareSoftware
{

    PREPARE_RC=0
    NO_OF_ZIPS=$(2>/dev/null ls "${INSTALLER_DOWNLOADS_SOFTWARE_DIRECTORY}"/*.zip | wc -l)
    
    # cleanTempAndUnzip calls isInfluxDBNeeded which will set FLEXDEPLOY_VERSION
    # FLEXDEPLOY_VERSION will be 6.5.0.0.150, so it contains 5 numbers.
    # it will always be lower case by here.
    if [ "${DESIRED_VERSION}" != "latest" ]
    then
        if [ "$NO_OF_ZIPS" -eq 1 ]
        then
            logPoint "Existing FlexDeploy zip found. Extracting it to verify the version."
            cleanTempAndUnzip ignore
            if [[ "${FLEXDEPLOY_VERSION}" == "${DESIRED_VERSION}"* ]]
            then
                logPoint "Existing FlexDeploy zip matched version ${DESIRED_VERSION}."
            elif variableEmpty "${FLEXDEPLOY_VERSION}"
            then
                logPoint "Removing corrupt FlexDeploy zip."
                rm -f "${INSTALLER_DOWNLOADS_SOFTWARE_DIRECTORY}"/*.zip > /dev/null 2>>"${LOG_FILE}"
            else
                logPoint "Existing FlexDeploy zip version ${FLEXDEPLOY_VERSION} did not match version ${DESIRED_VERSION}. Deleting it and attempting to download the correct version."
                rm -f "${INSTALLER_DOWNLOADS_SOFTWARE_DIRECTORY}"/*.zip > /dev/null 2>>"${LOG_FILE}"
            fi
        elif [ "$NO_OF_ZIPS" -ne 0 ]
        then
            logPoint "Multiple FlexDeploy zip files located. Deleting them and attempting to download the correct version."
            rm -f "${INSTALLER_DOWNLOADS_SOFTWARE_DIRECTORY}"/*.zip > /dev/null 2>>"${LOG_FILE}" 
        fi
        NO_OF_ZIPS=$(2>/dev/null ls "${INSTALLER_DOWNLOADS_SOFTWARE_DIRECTORY}"/*.zip | wc -l)
        if [ "$NO_OF_ZIPS" -eq 0 ]
        then
            logPoint "Downloading version ${DESIRED_VERSION} of FlexDeploy software from flexagon.com"

            # get the specified version of the software from the site
            wget -O "${INSTALLER_DOWNLOADS_SOFTWARE_DIRECTORY}"/"${TOMCAT_COMPLETE_ZIP_FILE_NAME}" "${FLEXDEPLOY_ALT_SOFTWARE_DOWNLOAD_URL}" --timeout=12 --tries=1 > /dev/null 2>&1
            DOWNLOAD_RC=$?
            if [ ${DOWNLOAD_RC} -ne 0 ]
            then
                # If the download fails, then delete the blank file that we get.
                rm -f "${INSTALLER_DOWNLOADS_SOFTWARE_DIRECTORY}"/"${TOMCAT_COMPLETE_ZIP_FILE_NAME}" > /dev/null 2>>"${LOG_FILE}" 
                generateFatalErrorMessage 121 "${DESIRED_VERSION}" "${FLEXDEPLOY_ALT_SOFTWARE_DOWNLOAD_URL}" "${INSTALLER_DOWNLOADS_SOFTWARE_DIRECTORY}"/"${TOMCAT_COMPLETE_ZIP_FILE_NAME}"
                exit ${DOWNLOAD_RC}
            fi
            cleanTempAndUnzip final
        fi
    else
        logPoint "Installing the latest FlexDeploy version was requested."
        setLatestVersion
        NO_INTERNET=$?
        if [ ${NO_INTERNET} -eq 0 ]
        then
            logPoint "The latest FlexDeploy version is ${LATEST_VERSION}."
        fi
        if [ "$NO_OF_ZIPS" -eq 1 ]
        then
            logPoint "Existing zip file found. Extracting to see if it matches the latest available version."
            cleanTempAndUnzip ignore
            
            if [ "${LATEST_VERSION}" != "0" ] && [[ "${FLEXDEPLOY_VERSION}" == "${LATEST_VERSION}"* ]]
            then
                logPoint "Existing FlexDeploy zip matched version ${LATEST_VERSION}."
            elif variableEmpty "${FLEXDEPLOY_VERSION}"
            then
                logPoint "Removing corrupt FlexDeploy zip."
                rm -f "${INSTALLER_DOWNLOADS_SOFTWARE_DIRECTORY}"/*.zip > /dev/null 2>>"${LOG_FILE}"
            elif [ "${LATEST_VERSION}" = "0" ]
            then
                logPoint "Existing FlexDeploy zip version ${FLEXDEPLOY_VERSION} was found. The latest version was not able to be checked from Flexagon.com. To continue the installer with version ${FLEXDEPLOY_VERSION}, press Y. To stop the installer press N."
                goOrStop
            else
                logPoint "Existing FlexDeploy zip version ${FLEXDEPLOY_VERSION} did not match version ${LATEST_VERSION}. Deleting it and attempting to download the correct version."
                rm -f "${INSTALLER_DOWNLOADS_SOFTWARE_DIRECTORY}"/*.zip > /dev/null 2>>"${LOG_FILE}"
            fi
        elif [ "$NO_OF_ZIPS" -ne 0 ]
        then
            logPoint "Multiple FlexDeploy zip files located. Deleting them and attempting to download the correct version."
            rm -f "${INSTALLER_DOWNLOADS_SOFTWARE_DIRECTORY}"/*.zip > /dev/null 2>>"${LOG_FILE}"
        fi
        NO_OF_ZIPS=$(2>/dev/null ls "${INSTALLER_DOWNLOADS_SOFTWARE_DIRECTORY}"/*.zip | wc -l)
        if [ "$NO_OF_ZIPS" -eq 0 ]
        then
            logPoint "No existing zips found. Downloading latest FlexDeploy software from flexagon.com"

            # get the latest version of the software from the site
            #No zip file, DESIRED_VERSION was latest, so attempt to download the latest.
            wget -O "${INSTALLER_DOWNLOADS_SOFTWARE_DIRECTORY}"/"${TOMCAT_COMPLETE_ZIP_FILE_NAME}" "${FLEXDEPLOY_SOFTWARE_DOWNLOAD_URL}" --timeout=12 --tries=1 > /dev/null 2>&1
            DOWNLOAD_RC=$?
            if [ ${DOWNLOAD_RC} -ne 0 ]
            then
                generateFatalErrorMessage 122 "${FLEXDEPLOY_SOFTWARE_DOWNLOAD_URL}" "${INSTALLER_DOWNLOADS_SOFTWARE_DIRECTORY}"/"${TOMCAT_COMPLETE_ZIP_FILE_NAME}"
                exit ${DOWNLOAD_RC}
            fi
            cleanTempAndUnzip final
        fi        
    fi

}

# function: cleanTempAndUnzip
#
# parameters
#    $1 - "ignore" will prevent fatal error message 133 from being thrown.
function cleanTempAndUnzip
{
    logPoint "Removing already unpacked FlexDeploy software"
    rm -rf "${INSTALLER_TEMP_SOFTWARE_DIRECTORY}" > /dev/null 2>>"${LOG_FILE}"
    PREPARE_RC=$?
    if [ ${PREPARE_RC} -ne 0 ]
    then
        generateFatalErrorMessage 120 "${INSTALLER_TEMP_SOFTWARE_DIRECTORY}"
    else
        setupMountPointArrays # reset free space after this as it has increased.
        logPoint "Checking for space to extract FlexDeploy software in ${INSTALLER_TEMP_SOFTWARE_DIRECTORY}"
        createDirectory "${INSTALLER_TEMP_SOFTWARE_DIRECTORY}"
        requireGbFree "${INSTALLER_TEMP_SOFTWARE_DIRECTORY}" 3

        logPoint "Unpackaging FlexDeploy software to ${INSTALLER_TEMP_SOFTWARE_DIRECTORY}"
        # unzip the download
        unzipper "${INSTALLER_DOWNLOADS_SOFTWARE_DIRECTORY}" "${INSTALLER_TEMP_SOFTWARE_DIRECTORY}" "$1"
        setupMountPointArrays # reset free space after this as it has decreased.
        PREPARE_RC=$?
        
        # following call will derive FLEXDEPLOY_VERSION from the zip file found.
        setZipFlexDeployVersion
        
        isInfluxDBNeeded
    fi
}



# function: testOrUpdateContextFile
#
# parameters
#    $1 Prereq Mode
#
# return
#    none
#
# function will configure the context.xml file based on database type, install/upgrade
# or the MAINTAIN_EXISTING_CONTEXT_FILE.  If maintain existing, the context file will be copied from
# the backup

function testOrUpdateContextFile
{
    # setup the source and target paths for the context file
    if isOracleDatabase
    then
        SOURCE_DIR=${INSTALLER_TEMP_SOFTWARE_DIRECTORY}/${APACHE_TOMCAT_FLEXDEPLOY_CONF}
        SOURCE_FILE=${CONTEXT_XML_ORACLE}
    else
        SOURCE_DIR=${INSTALLER_TEMP_SOFTWARE_DIRECTORY}/${APACHE_TOMCAT_FLEXDEPLOY_CONF}
        SOURCE_FILE=${CONTEXT_XML_POSTGRES}
    fi
    TARGET_DIR=${FLEXDEPLOY_HOME}/${APACHE_TOMCAT_FLEXDEPLOY_CONF}
    TARGET_FILE=${TOMCAT_CONTEXT_XML}

    if [ "${1}" -eq 1 ] && [ "${EXECUTION_STATE}" -eq "${UPGRADE}" ]
    then
        TARGET_DIR=${INSTALLER_TEMP_CONFIG_DIRECTORY}
    elif [ "${EXECUTION_STATE}" -eq "${UPGRADE}" ] && [ "${MAINTAIN_EXISTING_CONTEXT_FILE}" -eq 1 ]
    then
        SOURCE_DIR=${UPGRADE_BACKUP_TOMCAT_DIRECTORY}/${APACHE_TOMCAT_FLEXDEPLOY_CONF}
        SOURCE_FILE=${TOMCAT_CONTEXT_XML}
    fi

    # copy is a lcl function that handles mkdir and cp
    copy "${SOURCE_DIR}" "${SOURCE_FILE}" "${TARGET_DIR}" "${TARGET_FILE}"

    prompt "Enter FlexDeploy ${FLEXDEPLOY_DB_USER} password" "${FLEXDEPLOY_DB_PASSWORD}"
    FLEXDEPLOY_DB_PASSWORD=${PROMPT_VALUE}


    if [ "${MAINTAIN_EXISTING_CONTEXT_FILE}" -eq 0 ] || [ "${EXECUTION_STATE}" -eq "${INSTALL}" ]
    then
    # adjust the properties in the context.xml file based on the flexdeployinstaller_config.txt inputs
        sed -i "s|url=".*"|url=\"${DATABASE_URL}\"|g" "${TARGET_DIR}"/"${TARGET_FILE}"
        sed -i "s|user=".*"|user=\"${FLEXDEPLOY_DB_USER}\"|g" "${TARGET_DIR}"/"${TARGET_FILE}"

        # Need to regex escape the pw before trying to put it in the file.
        ESCAPED_FLEXDEPLOY_DB_PASSWORD=$(printf '%s\n' "$FLEXDEPLOY_DB_PASSWORD" | sed -e 's/[\/&]/\\&/g')
        sed -i "s|password=".*"|password=\"${ESCAPED_FLEXDEPLOY_DB_PASSWORD}\"|g" "${TARGET_DIR}"/"${TARGET_FILE}"
    fi

    if [ "${1}" -eq 1 ] && [ "${EXECUTION_STATE}" -eq "${UPGRADE}" ]
    then
        # check for user changes
        diff -Bbw "${FLEXDEPLOY_HOME}"/"${APACHE_TOMCAT_FLEXDEPLOY_CONF}"/"${TARGET_FILE}" "${INSTALLER_TEMP_CONFIG_DIRECTORY}"/"${TARGET_FILE}" > "${LOG_DATA_FILE}" 2>>"${LOG_FILE}"
        DIFF_RC=$?

        if [ ${DIFF_RC} -ne 0 ]
        then
            logPoint "MAINTAIN_EXISTING_CONTEXT_FILE is currently set to ${MAINTAIN_EXISTING_CONTEXT_FILE}"
            if [ "${MAINTAIN_EXISTING_CONTEXT_FILE}" -eq 1 ]
            then
                # adjust the < and > symbols in the file to be readable
                sed -i "s|^<|Your Config: |g" "${LOG_DATA_FILE}"
                sed -i "s|^>|Not applied: |g" "${LOG_DATA_FILE}"
                generateWarningErrorMessage 128 "context.xml"
            else
                # adjust the < and > symbols in the file to be readable
                sed -i "s|^<|Current Data: |g" "${LOG_DATA_FILE}"
                sed -i "s|^>|Future  Data: |g" "${LOG_DATA_FILE}"
                generateWarningErrorMessage 109 "context.xml"
            fi           
        fi
        rm -f "${INSTALLER_TEMP_CONFIG_DIRECTORY}"/"${TARGET_FILE}"
    fi
}

# function: testOrUpdateServerFile
#
# parameters
#    $1 prereq mode
#
# return
#    none
#
# function will configure the server.xml file based on install/upgrade
# or the MAINTAIN_EXISTING_SERVER_FILE.  If maintain existing, the server file will be copied from
# the backup

function testOrUpdateServerFile
{
    SOURCE_DIR=${INSTALLER_TEMP_SOFTWARE_DIRECTORY}/${APACHE_TOMCAT_FLEXDEPLOY_CONF}
    SOURCE_FILE=${TOMCAT_SERVER_XML}

    TARGET_DIR=${FLEXDEPLOY_HOME}/${APACHE_TOMCAT_FLEXDEPLOY_CONF}
    TARGET_FILE=${TOMCAT_SERVER_XML}

    if [ "${1}" -eq 1 ] && [ "${EXECUTION_STATE}" -eq "${UPGRADE}" ]
    then
        TARGET_DIR=${INSTALLER_TEMP_CONFIG_DIRECTORY}
    elif [ "${EXECUTION_STATE}" -eq "${UPGRADE}" ] && [ "${MAINTAIN_EXISTING_SERVER_FILE}" -eq 1 ]
    then
        SOURCE_DIR=${UPGRADE_BACKUP_TOMCAT_DIRECTORY}/${APACHE_TOMCAT_FLEXDEPLOY_CONF}
    fi

    # copy is a lcl function that handles mkdir and cp
    copy "${SOURCE_DIR}" "${SOURCE_FILE}" "${TARGET_DIR}" "${TARGET_FILE}"

    # adjust the FlexDeploy port if needed
    if [ "${MAINTAIN_EXISTING_SERVER_FILE}" -eq 0 ] || [ "${EXECUTION_STATE}" -eq "${INSTALL}" ]
    then
      sed -i "s|8000|${FLEXDEPLOY_PORT}|g" "${TARGET_DIR}"/"${TARGET_FILE}"
    fi

    if [ "${1}" -eq 1 ] && [ "${EXECUTION_STATE}" -eq "${UPGRADE}" ]
    then
        # check for user changes
        diff -Bbw "${FLEXDEPLOY_HOME}"/"${APACHE_TOMCAT_FLEXDEPLOY_CONF}"/"${TARGET_FILE}" "${INSTALLER_TEMP_CONFIG_DIRECTORY}"/"${TARGET_FILE}" > "${LOG_DATA_FILE}" 2>>"${LOG_FILE}"
        DIFF_RC=$?

        if [ ${DIFF_RC} -ne 0 ]
        then
            
            
            logPoint "MAINTAIN_EXISTING_SERVER_FILE is currently set to ${MAINTAIN_EXISTING_SERVER_FILE}"
            if [ "${MAINTAIN_EXISTING_SERVER_FILE}" -eq 1 ]
            then
                # adjust the < and > symbols in the file to be readable
                sed -i "s|^<|Your Config: |g" "${LOG_DATA_FILE}"
                sed -i "s|^>|Not applied: |g" "${LOG_DATA_FILE}"
                generateWarningErrorMessage 127 "server.xml"
            else
                # adjust the < and > symbols in the file to be readable
                sed -i "s|^<|Current Data: |g" "${LOG_DATA_FILE}"
                sed -i "s|^>|Future  Data: |g" "${LOG_DATA_FILE}"
                generateWarningErrorMessage 111 "server.xml"
            fi
            
        fi
        rm -f "${INSTALLER_TEMP_CONFIG_DIRECTORY}"/"${TARGET_FILE}"
    fi
}

# function: backupCurrentInstallation
#
# parameters
#    none
#
# return
#    none
#
# function will backup the current installion if this is for an upgrade and not in prereq mode (running prerequisites without install),
# also makes sure that it is only backed up once per day

function backupCurrentInstallation
{
    # docker and weblogic don't utilize libext so we need to source the driver from the temp location, just like prereqs
	
    # since an upgrade, backup the working installation, only do this if not PREREQ_MODE and not DOCKER_INSTALL
    if [ "${EXECUTION_STATE}" -eq "${UPGRADE}" ] && [ "${PREREQ_MODE}" -eq 0 ] && [ "${DOCKER_INSTALL}" -eq 0 ]
    then
        # --------------- BACKUP apache-tomcat-flexdeploy ---------------
        # check if a backup already exists for the version
        if [ -d "${UPGRADE_BACKUP_TOMCAT_DIRECTORY}"/"${APACHE_TOMCAT_FLEXDEPLOY}" ]
        then
            logPoint "A backup already exists (${UPGRADE_BACKUP_TOMCAT_DIRECTORY}/${APACHE_TOMCAT_FLEXDEPLOY}), no backup created"
        else
            logPoint "Deleting the temp folder under ${FLEXDEPLOY_HOME}"/"${APACHE_TOMCAT_FLEXDEPLOY} before back up"
            rm -rf "${FLEXDEPLOY_HOME}"/"${APACHE_TOMCAT_FLEXDEPLOY}"/temp
            logPoint "Deleted temp folder from ${FLEXDEPLOY_HOME}"/"${APACHE_TOMCAT_FLEXDEPLOY}"
            
            # copy the current tomcat installation
            logPoint "Backing up Tomcat installation from ${FLEXDEPLOY_HOME}/${APACHE_TOMCAT_FLEXDEPLOY} to ${UPGRADE_BACKUP_TOMCAT_DIRECTORY}"
            cp -r "${FLEXDEPLOY_HOME}"/"${APACHE_TOMCAT_FLEXDEPLOY}" "${UPGRADE_BACKUP_TOMCAT_DIRECTORY}"
		fi
        
        # printing details of libext folder for our reference. could be useful during support
        logPoint "listing of files in ${UPGRADE_BACKUP_TOMCAT_DIRECTORY}/${APACHE_TOMCAT_FLEXDEPLOY}"
        ls -ltr "${UPGRADE_BACKUP_TOMCAT_DIRECTORY}/${APACHE_TOMCAT_FLEXDEPLOY}" | tee -a "${LOG_FILE}"
        logPoint "listing of files in ${UPGRADE_BACKUP_TOMCAT_DIRECTORY}/${APACHE_TOMCAT_FLEXDEPLOY_LIBEXT}"
        ls -ltr "${UPGRADE_BACKUP_TOMCAT_DIRECTORY}/${APACHE_TOMCAT_FLEXDEPLOY_LIBEXT}" | tee -a "${LOG_FILE}"
        
        # --------------- BACKUP apache-tomcat-flexdeploy ---------------
    fi
    
    # do the following block for upgrade AND upgrade_WLS, but not install. and not PREREQ_MODE
    if [ "${EXECUTION_STATE}" -ne "${INSTALL}" ] && [ "${PREREQ_MODE}" -eq 0 ]
    then

        # save current directory so we can restore back
        CURRENT_PWD=$(pwd)

        # --------------- BACKUP Artifacts Repository ---------------
        if [ -f "${UPGRADE_BACKUP_ARTIFACTS_DIRECTORY}"/artifacts.tar.gz ]
        then
            logPoint "A backup already exists (${UPGRADE_BACKUP_ARTIFACTS_DIRECTORY}/artifacts.tar.gz), no backup created"
        else 
            if [ "${IGNORE_ARTIFACT_BACKUP}" -eq 0 ]
            then
                # must cd in to repository folder, so we can restore in same manner without any additional folders
                cd "${ARTIFACT_REPOSITORY_DIRECTORY}"
                ARTIFACTS_EXIST=$?
                if [ ${ARTIFACTS_EXIST} -eq 0 ]
                then
                    # copy the artifacts
                    logPoint "Backing up the artifacts directory from ${ARTIFACT_REPOSITORY_DIRECTORY} to ${UPGRADE_BACKUP_ARTIFACTS_DIRECTORY}, may take some time"
                    tar -czf "${UPGRADE_BACKUP_ARTIFACTS_DIRECTORY}"/artifacts.tar.gz *  >> "${LOG_FILE}" 2>&1
                    T_RC=$?
                    if [ ${T_RC} -ne 0 ]
                    then
                        # delete the artifacts backup file.
                        rm -f "${UPGRADE_BACKUP_ARTIFACTS_DIRECTORY}"/artifacts.tar.gz >> "${LOG_FILE}" 2>&1						
                        generateFatalErrorMessage 135 "${ARTIFACT_REPOSITORY_DIRECTORY}" "${UPGRADE_BACKUP_ARTIFACTS_DIRECTORY}"
                    fi
                else
                    logPoint "No artifacts found at ${ARTIFACT_REPOSITORY_DIRECTORY}."
                    generateFatalErrorMessage 137 "${ARTIFACT_REPOSITORY_DIRECTORY}"
                fi
            else
                logPoint "Skipping backing up the artifacts directory as requested."
            fi
        fi
        #Don't check the size again for this.
        SKIP_ARTIFACT_SIZE_CHECK=1
        # --------------- BACKUP Artifacts Repository ---------------
        
        # --------------- BACKUP plugins folder  ---------------
        if [ -f "${UPGRADE_BACKUP_PLUGINS_DIRECTORY}"/plugins.tar.gz ]
        then
            logPoint "A backup already exists (${UPGRADE_BACKUP_PLUGINS_DIRECTORY}/plugins.tar.gz), no backup created"
        else        
            logPoint "Backing up the plugins directory from ${WORKING_DIRECTORY}/plugins to ${UPGRADE_BACKUP_PLUGINS_DIRECTORY}, may take some time"

            # must cd in to plugins folder, so we can restore in same manner without any additional folders
            cd "${WORKING_DIRECTORY}"/plugins
            PLUGINS_EXIST=$?
            if [ ${PLUGINS_EXIST} -eq 0 ]
            then
                tar -czf "${UPGRADE_BACKUP_PLUGINS_DIRECTORY}"/plugins.tar.gz * >> "${LOG_FILE}" 2>&1
                T_RC=$?
                if [ ${T_RC} -ne 0 ]
                then 
                    generateFatalErrorMessage 135 "${WORKING_DIRECTORY}"/plugins "${UPGRADE_BACKUP_PLUGINS_DIRECTORY}"
                fi
            else
                logPoint "No plugins found at ${WORKING_DIRECTORY}/plugins."
                logPoint "This is expected for older versions of FlexDeploy which stored them in the database instead."
            fi
        fi
        #don't check the size again after this.
        SKIP_PLUGIN_SIZE_CHECK=1
        # --------------- BACKUP plugins folder  ---------------

        # go back to original directory
        cdOrFail "${CURRENT_PWD}"

        # --------------- BACKUP database ---------------
        if [ "${IGNORE_DB_BACKUP}" -eq 0 ]
        then
            # backup the database schemas
            logPoint "Backing up the database schemas, this may take some time"

            prompt "Enter database system password" "${SYSTEM_DB_PASSWORD}"
            SYSTEM_DB_PASSWORD=${PROMPT_VALUE}

            if variableEmpty "${BACKUP_DATABASE_DIRECTORY}"
            then
                "${FLEXDEPLOY_JAVA_HOME}"/bin/java -cp "${DB_UTIL_CLASSPATH}" flexagon.fd.installer.DatabaseManagement -OPERATION BACKUP -URL "${DATABASE_URL}" -DB_USERNAME "${SYSTEM_DB_USER}" -DB_PASSWORD "${SYSTEM_DB_PASSWORD}" -LOG_FILE "${LOG_FILE}"
            else
                "${FLEXDEPLOY_JAVA_HOME}"/bin/java -cp "${DB_UTIL_CLASSPATH}" flexagon.fd.installer.DatabaseManagement -OPERATION BACKUP -URL "${DATABASE_URL}" -DB_USERNAME "${SYSTEM_DB_USER}" -DB_PASSWORD "${SYSTEM_DB_PASSWORD}" -BACKUP_DIR "${BACKUP_DATABASE_DIRECTORY}" -LOG_FILE "${LOG_FILE}"
            fi
            DB_UTIL_RC=$?
            handleDbError ${DB_UTIL_RC} "${SYSTEM_DB_USER}"
        else
            logPoint "No database backup created as requested."
        fi
        # --------------- BACKUP database ---------------
	fi
}

# function: manageSetEnvOverride
#
# parameters
#    none
#
# return
#    none
#
# function will manage the setupoverride file with configuration details

function manageSetEnvOverride
{
    # always set it as exists by default
	SETENVOVERRIDE_FILE_EXISTS=1
	
    # use the setenv.sh and setenvoverride.sh script if there is one present in the backups
    if [ -f "${UPGRADE_BACKUP_FLEXDEPLOY_TOMCAT_BIN_DIRECTORY}"/"${SETENVOVERRIDE_FILE_NAME}" ]
    then
        cp "${UPGRADE_BACKUP_FLEXDEPLOY_TOMCAT_BIN_DIRECTORY}"/"${SETENVOVERRIDE_FILE_NAME}" "${FLEXDEPLOY_TOMCAT_BIN_DIRECTORY}"/"${SETENVOVERRIDE_FILE_NAME}" > /dev/null 2>>"${LOG_FILE}"
        if [ "${INFLUX_DB_NEEDED}" -eq 1 ]
        then
            # since this needs influx and is an upgrade, the FD_INFLUX_CONFIG property may not be set in the setenvoverride.sh file
            # because we copy from the backup which is probably an older version
            if ! grep FD_INFLUX_CONFIG "${FLEXDEPLOY_TOMCAT_BIN_DIRECTORY}"/"${SETENVOVERRIDE_FILE_NAME}" > /dev/null 2>>"${LOG_FILE}"
            then
                # add the line that is needed for the influx upgrade
                echo >> "${FLEXDEPLOY_TOMCAT_BIN_DIRECTORY}"/"${SETENVOVERRIDE_FILE_NAME}"
                echo "FD_INFLUX_CONFIG=${INFLUX_DB_CONFIG_DIRECTORY}/fdinflux.config" >> "${FLEXDEPLOY_TOMCAT_BIN_DIRECTORY}"/"${SETENVOVERRIDE_FILE_NAME}"
            fi
            # the setenvoverride java home should match the one we are using. If not, update it now.
            if ! grep FLEXDEPLOY_JAVA_HOME "${FLEXDEPLOY_TOMCAT_BIN_DIRECTORY}"/"${SETENVOVERRIDE_FILE_NAME}" > /dev/null 2>>"${LOG_FILE}"
            then
                logPoint "Updating the Java Home for FlexDeploy."
                if ! grep "^JAVA_HOME=.*" "${FLEXDEPLOY_TOMCAT_BIN_DIRECTORY}"/"${SETENVOVERRIDE_FILE_NAME}" > /dev/null 2>>"${LOG_FILE}"
                then
                  # if JAVA_HOME was not set before, set it now.
                  echo >> "${FLEXDEPLOY_TOMCAT_BIN_DIRECTORY}"/"${SETENVOVERRIDE_FILE_NAME}"
                  echo "JAVA_HOME=${FLEXDEPLOY_JAVA_HOME}" >> "${FLEXDEPLOY_TOMCAT_BIN_DIRECTORY}"/"${SETENVOVERRIDE_FILE_NAME}"

                else
                  # if it was, update it.
                  sed -i "s|^JAVA_HOME=.*|JAVA_HOME=${FLEXDEPLOY_JAVA_HOME}|g" "${FLEXDEPLOY_TOMCAT_BIN_DIRECTORY}"/"${SETENVOVERRIDE_FILE_NAME}"
                fi

            fi
        fi
    else
        # if there is no file or this is a new installation, use the files from the download and adjust per the flexdeployinstaller_config.txt file
        if [ "${EXECUTION_STATE}" -eq "${UPGRADE}" ] && [ ! -f "${UPGRADE_BACKUP_FLEXDEPLOY_TOMCAT_BIN_DIRECTORY}"/"${SETENVOVERRIDE_FILE_NAME}" ]
        then
            generateWarningErrorMessage 112

            # set a flag so that we can not start the server if the file didn't exist and explain how to fix this
            SETENVOVERRIDE_FILE_EXISTS=0
    	fi

        # adjust the file per the flexdeployinstaller_config.txt file
        sed -i "s|FD_APPLICATION_DIRECTORY=".*"|FD_APPLICATION_DIRECTORY=${WORKING_DIRECTORY}|g" "${FLEXDEPLOY_TOMCAT_BIN_DIRECTORY}"/"${SETENVOVERRIDE_FILE_NAME}"
        sed -i "s|FD_REPOSITORY_DIRECTORY=".*"|FD_REPOSITORY_DIRECTORY=${ARTIFACT_REPOSITORY_DIRECTORY}|g" "${FLEXDEPLOY_TOMCAT_BIN_DIRECTORY}"/"${SETENVOVERRIDE_FILE_NAME}"
        sed -i "s|FD_INFLUX_CONFIG=".*"|FD_INFLUX_CONFIG=${INFLUX_DB_CONFIG_DIRECTORY}/fdinflux.config|g" "${FLEXDEPLOY_TOMCAT_BIN_DIRECTORY}"/"${SETENVOVERRIDE_FILE_NAME}"
        sed -i "s|#JAVA_HOME=".*"|JAVA_HOME=${FLEXDEPLOY_JAVA_HOME}|g" "${FLEXDEPLOY_TOMCAT_BIN_DIRECTORY}"/"${SETENVOVERRIDE_FILE_NAME}"
    fi
}

# function: manageDatabaseDownloads
#
# parameters
#    none
#
# return
#    none
#
# function will make sure the appropriate database driver is configured
# and setup up any necessary wallets

function manageDatabaseDownloads
{
    # only need to adjust if an oracle database
    
    if isOracleDatabase
    then
        logPoint "Preparing the new oracle database driver"
    fi
    
    # remove any existing expanded files
    # delete everything in INSTALLER_TEMP_DRIVER_DIRECTORY and then unzip again.
    rm -rf "${INSTALLER_TEMP_DRIVER_DIRECTORY}"/* >> "${LOG_FILE}" 2>&1

    # un tar the downloaded oracle driver
    tar xf "${FLEXDEPLOY_INSTALLER_HOME}"/scripts/jars/ojdbc8*.gz -C "${INSTALLER_TEMP_DRIVER_DIRECTORY}" > /dev/null 2>>"${LOG_FILE}"
    T_RC=$?
    if [ ${T_RC} -ne 0 ]
    then 
        generateFatalErrorMessage 134 "${FLEXDEPLOY_INSTALLER_HOME}"/scripts/jars/ojdbc8*.gz  "${INSTALLER_TEMP_DRIVER_DIRECTORY}"
    fi

    # check to see if a database wallet has been downloaded
    ls "${INSTALLER_DOWNLOADS_WALLETS_DIRECTORY}"/*.zip > /dev/null 2>&1
    if [ $? -eq 0 ]
    then
        NUM_OF_ZIPS=$(ls "${INSTALLER_DOWNLOADS_WALLETS_DIRECTORY}"/*.zip | wc -l)
    else
        # no zip files in wallet download
        NUM_OF_ZIPS=0
    fi

    if [ "${NUM_OF_ZIPS}" -gt 0 ]
    then
        logPoint "Preparing the oracle wallet files"
        for WALLET in $(ls "${INSTALLER_DOWNLOADS_WALLETS_DIRECTORY}"/*.zip)
        do
            # get the filename so we can use it as a base folder
            WALLET_NAME=$(basename "${WALLET}" | cut -f1 -d'.')
            logPoint "Preparing wallet ${WALLET} for use."
            # remove any existing files and unzip the wallet
            rm -rf "${FLEXDEPLOY_DATABASE_WALLET_DIRECTORY}"/"${WALLET_NAME}" > /dev/null 2>>"${LOG_FILE}"
            unzip "${WALLET}" -d "${FLEXDEPLOY_DATABASE_WALLET_DIRECTORY}/${WALLET_NAME}" > /dev/null 2>>"${LOG_FILE}"
            logPoint "Wallet ${WALLET_NAME} prepared for use."
        done
    fi

    # download the influxdb jar if internet is open and it is needed for the release
    if [ "${INFLUX_DB_NEEDED}" -eq 1 ]
    then
        if [ "${FLEXDEPLOY_VERSION#1}" != "$FLEXDEPLOY_VERSION" ]
        then
          INFLUX_DB_DOWNLOAD_URL=${INFLUX_DB_DOWNLOAD_URL_10}
          INFLUX_CLIENT_DOWNLOAD_URL=${INFLUX_CLIENT_DOWNLOAD_URL_10}
          INFLUX_DB_FILE_NAME=${INFLUX_DB_FILE_NAME_10}
          INFLUX_DB_CLIENT_FILE_NAME=${INFLUX_DB_CLIENT_FILE_NAME_10}
        fi
        logPoint "Preparing the influx database and client files"
        FAILED=0
        createDirectory "${INSTALLER_DOWNLOADS_INFLUXDB_DIRECTORY}"
        if [ ! -f "${INSTALLER_DOWNLOADS_INFLUXDB_DIRECTORY}"/"${INFLUX_DB_FILE_NAME}" ]
        then
            logPoint "Downloading influx database."
            wget -O "${INSTALLER_DOWNLOADS_INFLUXDB_DIRECTORY}"/"${INFLUX_DB_FILE_NAME}" "${INFLUX_DB_DOWNLOAD_URL}" --timeout=12 --tries=1 > /dev/null 2>&1
            T_RC=$?
            if [ ${T_RC} -ne 0 ]
            then 
                # If the download fails, then delete the blank file that we get.
                rm -f "${INSTALLER_DOWNLOADS_INFLUXDB_DIRECTORY}"/"${INFLUX_DB_FILE_NAME}" > /dev/null 2>>"${LOG_FILE}"
                generateFatalErrorMessage 136 "${INFLUX_DB_DOWNLOAD_URL}" "${INSTALLER_DOWNLOADS_INFLUXDB_DIRECTORY}"/"${INFLUX_DB_FILE_NAME}"
                FAILED=1
            fi
        fi

        if [ ! -f "${INSTALLER_DOWNLOADS_INFLUXDB_DIRECTORY}"/"${INFLUX_DB_CLIENT_FILE_NAME}" ]
        then
            logPoint "Downloading influx client."
            wget -O "${INSTALLER_DOWNLOADS_INFLUXDB_DIRECTORY}"/"${INFLUX_DB_CLIENT_FILE_NAME}" "${INFLUX_CLIENT_DOWNLOAD_URL}" --timeout=12 --tries=1 > /dev/null 2>&1
            T_RC=$?
            if [ ${T_RC} -ne 0 ]
            then 
                # If the download fails, then delete the blank file that we get.
                rm -f "${INSTALLER_DOWNLOADS_INFLUXDB_DIRECTORY}"/"${INFLUX_DB_CLIENT_FILE_NAME}" > /dev/null 2>>"${LOG_FILE}"
                generateFatalErrorMessage 136 "${INFLUX_CLIENT_DOWNLOAD_URL}" "${INSTALLER_DOWNLOADS_INFLUXDB_DIRECTORY}"/"${INFLUX_DB_CLIENT_FILE_NAME}"
                FAILED=1
            fi
        fi

        if [ ${FAILED} -eq 0 ]
        then 
            # untar the downloaded/
            logPoint "Expanding influx database software"
            tar xf "${INSTALLER_DOWNLOADS_INFLUXDB_DIRECTORY}"/"${INFLUX_DB_FILE_NAME}" --strip-components 1 -C "${INSTALLER_TEMP_INFLUXDB_DIRECTORY}"
                T_RC=$?
                if [ ${T_RC} -ne 0 ]
                then 
                    generateFatalErrorMessage 134 "${INSTALLER_DOWNLOADS_INFLUXDB_DIRECTORY}"/"${INFLUX_DB_FILE_NAME}"  "${INSTALLER_TEMP_INFLUXDB_DIRECTORY}"
                fi
            tar xf "${INSTALLER_DOWNLOADS_INFLUXDB_DIRECTORY}"/"${INFLUX_DB_CLIENT_FILE_NAME}" --strip-components 1 -C "${INSTALLER_TEMP_INFLUXDB_DIRECTORY}"
                T_RC=$?
                if [ ${T_RC} -ne 0 ]
                then 
                    generateFatalErrorMessage 134 "${INSTALLER_DOWNLOADS_INFLUXDB_DIRECTORY}"/"${INFLUX_DB_CLIENT_FILE_NAME}" "${INSTALLER_TEMP_INFLUXDB_DIRECTORY}"
                fi
        fi
    fi
}

# function: manageDatabaseObjects
#
# parameters
#    none
#
# return
#    none
#
# function will make sure the appropriate database driver is configured

function manageDatabaseObjects
{
    # need to make calls to either create the schemas (install) or update the schemas (upgrade)
    if [ "${EXECUTION_STATE}" -eq "${INSTALL}" ]
    then
        logPoint "Creating the database objects"

        prompt "Enter FlexDeploy ${FLEXDEPLOY_DB_USER} password" "${FLEXDEPLOY_DB_PASSWORD}"
        FLEXDEPLOY_DB_PASSWORD=${PROMPT_VALUE}

        prompt "Enter database system password" "${SYSTEM_DB_PASSWORD}"
        SYSTEM_DB_PASSWORD=${PROMPT_VALUE}

		"${FLEXDEPLOY_JAVA_HOME}"/bin/java -cp "${DB_UTIL_CLASSPATH}" flexagon.fd.installer.DatabaseManagement -OPERATION INSTALL -URL "${DATABASE_URL}" -DB_USERNAME "${SYSTEM_DB_USER}" -DB_PASSWORD "${SYSTEM_DB_PASSWORD}" -INSTALL_DIR "${INSTALLER_TEMP_SOFTWARE_DIRECTORY}" -SCHEMA_PASSWORD "${FLEXDEPLOY_DB_PASSWORD}" -LOG_FILE "${LOG_FILE}"	 -FD_ADMIN "${FLEXDEPLOY_DB_USER}"
	    DB_UTIL_RC=$?
	else
		logPoint "Updating the database objects"

        prompt "Enter FlexDeploy ${FLEXDEPLOY_DB_USER} password" "${FLEXDEPLOY_DB_PASSWORD}"
        FLEXDEPLOY_DB_PASSWORD=${PROMPT_VALUE}

        prompt "Enter database system password" "${SYSTEM_DB_PASSWORD}"
        SYSTEM_DB_PASSWORD=${PROMPT_VALUE}

		"${FLEXDEPLOY_JAVA_HOME}"/bin/java -cp "${DB_UTIL_CLASSPATH}" flexagon.fd.installer.DatabaseManagement -OPERATION UPGRADE -URL "${DATABASE_URL}" -DB_USERNAME "${SYSTEM_DB_USER}" -DB_PASSWORD "${SYSTEM_DB_PASSWORD}" -INSTALL_DIR "${INSTALLER_TEMP_SOFTWARE_DIRECTORY}" -SCHEMA_PASSWORD "${FLEXDEPLOY_DB_PASSWORD}" -LOG_FILE "${LOG_FILE}" -FD_ADMIN "${FLEXDEPLOY_DB_USER}"
	        DB_UTIL_RC=$?
	fi
	handleDbError ${DB_UTIL_RC}
	return ${DB_UTIL_RC}
}

# PRE 10.0 - use same code as before, with same influx versions as before.
function manageInfluxDB
{
     logPoint "Installing or upgrading influx database binaries in ${INFLUX_DB_DIRECTORY}."
     cp "${INSTALLER_TEMP_INFLUXDB_DIRECTORY}"/influx* "${INFLUX_DB_DIRECTORY}"
     manageInfluxDBCommon
}

# 10.0+ - use new code as before, with same influx versions as before.
function manageInfluxDB10
{
    logPoint "Installing or upgrading influx database binaries in ${INFLUX_DB_DIRECTORY}."
    cp "${INSTALLER_TEMP_INFLUXDB_DIRECTORY}"/influx "${INFLUX_DB_DIRECTORY}"
    cp "${INSTALLER_TEMP_INFLUXDB_DIRECTORY}"/usr/bin/influxd "${INFLUX_DB_DIRECTORY}"
    manageInfluxDBCommon
    
}

function manageInfluxDBCommon
{
  if [ "${INFLUX_DB_NEEDED}" -eq 1 ] && [ ! -d "${INFLUX_DB_VSM_DATA_DIRECTORY}"/data/engine/wal ]
    then
        logPoint "Configuring influxDB."
        cp "${INSTALLER_TEMP_SOFTWARE_DIRECTORY}"/influxdb/config/* "${INFLUX_DB_CONFIG_DIRECTORY}"
            sed -i "s|bolt-path: ".*"|bolt-path: ${INFLUX_DB_VSM_DATA_DIRECTORY}/data/influxd.bolt|g"  "${INFLUX_DB_CONFIG_DIRECTORY}"/config.yaml
            sed -i "s|engine-path: ".*"|engine-path: ${INFLUX_DB_VSM_DATA_DIRECTORY}/data/engine|g"  "${INFLUX_DB_CONFIG_DIRECTORY}"/config.yaml
            sed -i "s|sqlite-path: ".*"|sqlite-path: ${INFLUX_DB_VSM_DATA_DIRECTORY}/data/influxd.sqlite|g"  "${INFLUX_DB_CONFIG_DIRECTORY}"/config.yaml

            sed -i "s|INFLUXDB_URL=".*"|INFLUXDB_URL=${INFLUX_DB_URL}|g"  "${INFLUX_DB_CONFIG_DIRECTORY}"/fdinflux.config
            sed -i "s|INFLUXDB_BUCKET=".*"|INFLUXDB_BUCKET=${INFLUX_DB_BUCKET}|g"  "${INFLUX_DB_CONFIG_DIRECTORY}"/fdinflux.config
            sed -i "s|INFLUXDB_ORG=".*"|INFLUXDB_ORG=${INFLUX_DB_ORG}|g"  "${INFLUX_DB_CONFIG_DIRECTORY}"/fdinflux.config
    fi
}

# function: copyPlugin
#
# parameters
#    none
#
# return
#    none
#
# function will copy plugins from plugin temp directory from new install to working directory for auto upload process
function copyPlugins
{
    # copy plugin files
    logPoint "Copying new plugins from ${INSTALLER_TEMP_SOFTWARE_PLUGINS_DIRECTORY} to ${WORKING_DIRECTORY} for auto upload"
    cp -r  "${INSTALLER_TEMP_SOFTWARE_PLUGINS_DIRECTORY}" "${WORKING_DIRECTORY}"
}


# function: installFlexDeployDocker
#
# parameters
#    none
#
# return
#    0 - software installed
#    1 - install failed
#
# function will start the installation of the downloaded software
function installFlexDeployDocker
{
    # DOCKER INSTALL
    if [ "${EXECUTION_STATE}" -ne "${INSTALL}" ]
    then
        logPoint "Beginning Docker Upgrade."
    
        deleteDockerContainer
    else
        logPoint "Beginning Docker Install."
    fi
    
    # Manage database objects. Execute install or migration scripts
    manageDatabaseObjects
    
    # Copy plugins to working directory
    copyPlugins
    
    # Pull docker image based on target version
    pullDockerImage
    
    # Run docker container with updated image
    runDockerContainer
    
    logPoint "Docker Install Complete"
}

# function: installFlexDeploy
#
# parameters
#    none
#
# return
#    0 - software installed
#    1 - install failed
#
# function will start the installation of the downloaded software

function installFlexDeploy
{
    #Database first so that if it fails it won't try to upgrade next time.
    manageDatabaseObjects
    #Do this for install and upgrade_wls.
    if [ "${EXECUTION_STATE}" -ne "${UPGRADE}" ]
    then
        logPoint "Copying new software to ${FLEXDEPLOY_HOME}"
        cp -r "${INSTALLER_TEMP_SOFTWARE_DIRECTORY}"/* "${FLEXDEPLOY_HOME}" || exitFatalErrorMessage 144 "${FLEXDEPLOY_HOME}"

        # remove these directories to keep  the server folder clean, let them get added back based on the flexdeployinstaller_config file
        rm -rf "${FLEXDEPLOY_HOME}"/database >> "${LOG_FILE}" 2>&1 || exitFatalErrorMessage 143 "${FLEXDEPLOY_HOME}"/database
        rm -rf "${FLEXDEPLOY_HOME}"/application >> "${LOG_FILE}" 2>&1 || exitFatalErrorMessage 143 "${FLEXDEPLOY_HOME}"/application
        rm -rf "${FLEXDEPLOY_HOME}"/artifacts >> "${LOG_FILE}" 2>&1 || exitFatalErrorMessage 143 "${FLEXDEPLOY_HOME}"/artifacts
        
        # just in case the default directories are being used, lets make sure they are created
        createDirectory "${WORKING_DIRECTORY}"
        createDirectory "${ARTIFACT_REPOSITORY_DIRECTORY}" 
    else
        logPoint "Removing ${FLEXDEPLOY_HOME}/${APACHE_TOMCAT_FLEXDEPLOY} folder"
        rm -rf "${FLEXDEPLOY_HOME}"/"${APACHE_TOMCAT_FLEXDEPLOY}" >> "${LOG_FILE}" 2>&1 || tomcatDirDeleteFailedDuringUpgrade
        
        logPoint "Copying new apache-tomcat-flexdeploy software from ${INSTALLER_TEMP_SOFTWARE_DIRECTORY}/${APACHE_TOMCAT_FLEXDEPLOY} to ${FLEXDEPLOY_HOME}"

        cp -r "${INSTALLER_TEMP_SOFTWARE_DIRECTORY}"/"${APACHE_TOMCAT_FLEXDEPLOY}" "${FLEXDEPLOY_HOME}" || exitFatalErrorMessage 144 "${FLEXDEPLOY_HOME}"
        # due to needing this line above for the java calls, the remove 2 lines above cleans out all file systems so after the rm
        # we need to re-copy this libext so this is duplicated for a reason.  We want the database work to be one of the first things done, do not remove
        stat -t "${UPGRADE_BACKUP_TOMCAT_DIRECTORY}"/"${APACHE_TOMCAT_FLEXDEPLOY_LIBEXT}"/* >/dev/null 2>&1 && (cp "${UPGRADE_BACKUP_TOMCAT_DIRECTORY}"/"${APACHE_TOMCAT_FLEXDEPLOY_LIBEXT}"/* "${FLEXDEPLOY_HOME}"/"${APACHE_TOMCAT_FLEXDEPLOY_LIBEXT}" || exitFatalErrorMessage 144 "${FLEXDEPLOY_HOME}"/"${APACHE_TOMCAT_FLEXDEPLOY_LIBEXT}")
        
        # copy all files from the base folder (scripts, ...) and not directories so don't want a -r on this one,
        # not dumping to the logs because error indicates to add the -r and not on this one.
        cp "${INSTALLER_TEMP_SOFTWARE_DIRECTORY}"/* "${FLEXDEPLOY_HOME}" > /dev/null 2>&1
    fi
    
    # copy plugin files to the configured directory
    copyPlugins

    #If it starts with 1, (10,11) then we will use the new influx. otherwise, old.
    if [ "${FLEXDEPLOY_VERSION#1}" != "$FLEXDEPLOY_VERSION" ]
    then
      manageInfluxDB10
    else
      manageInfluxDB
    fi
    
    # decided to copy Oracle driver in libext for all customers and remove lib/ojdbc*.jar for all customers even PostgreSQL
    if [ "${EXECUTION_STATE}" -eq "${UPGRADE}" ]
    then
        logPoint "Updating oracle driver jars in ${FLEXDEPLOY_HOME}/${APACHE_TOMCAT_FLEXDEPLOY_LIBEXT} with the version shipped with the installer."
        stat -t "${FLEXDEPLOY_HOME}"/"${APACHE_TOMCAT_FLEXDEPLOY_LIBEXT}"/*ojdbc6*.jar >/dev/null 2>&1 && logPoint "Removing any old ojdbc jars from ${FLEXDEPLOY_HOME}/${APACHE_TOMCAT_FLEXDEPLOY_LIBEXT}"
        stat -t "${FLEXDEPLOY_HOME}"/"${APACHE_TOMCAT_FLEXDEPLOY_LIBEXT}"/*ojdbc7*.jar >/dev/null 2>&1 && logPoint "Removing any old ojdbc jars from ${FLEXDEPLOY_HOME}/${APACHE_TOMCAT_FLEXDEPLOY_LIBEXT}"
        rm -f "${FLEXDEPLOY_HOME}"/"${APACHE_TOMCAT_FLEXDEPLOY_LIBEXT}"/*ojdbc*.jar 2>/dev/null        
    fi
    
    # before copy make sure we are able to write the jars in libext folder
    chmod u+w ${FLEXDEPLOY_HOME}/${APACHE_TOMCAT_FLEXDEPLOY_LIBEXT}/*
    cp "${INSTALLER_TEMP_DRIVER_DIRECTORY}"/*/*.jar "${FLEXDEPLOY_HOME}"/"${APACHE_TOMCAT_FLEXDEPLOY_LIBEXT}" >> "${LOG_FILE}" 2>&1 || exitFatalErrorMessage 150

    # remove the current ojdbc jars
    stat -t "${FLEXDEPLOY_HOME}"/"${APACHE_TOMCAT_FLEXDEPLOY_LIB}"/*ojdbc*.jar >/dev/null 2>&1 && logPoint "Removing ojdbc driver files from ${FLEXDEPLOY_HOME}/${APACHE_TOMCAT_FLEXDEPLOY_LIB}."
    rm -f "${FLEXDEPLOY_HOME}"/"${APACHE_TOMCAT_FLEXDEPLOY_LIB}"/*ojdbc*.jar >> "${LOG_FILE}" 2>&1
     
    logPoint "Updating the context.xml file for Tomcat"
    testOrUpdateContextFile 0

    logPoint "Updating the server.xml file for Tomcat"
    testOrUpdateServerFile 0

    logPoint "Managing setenvoverride"
    manageSetEnvOverride

    logPoint "Updating script permissions"
    chmod +x "${FLEXDEPLOY_HOME}"/"${APACHE_TOMCAT_FLEXDEPLOY_BIN}"/*.sh
    chmod +x "${FLEXDEPLOY_HOME}"/*.sh

}

# function: startFlexDeploy
#
# parameters
#    none
#
# return
#    none
#
# function will start the tomcat server

function startFlexDeploy
{
    # start the tomcat server
    CURRENT_PWD=$(pwd)
    cdOrFail "${FLEXDEPLOY_HOME}/${APACHE_TOMCAT_FLEXDEPLOY_BIN}"

    # need to create the .token file before startup so the thread grabs the file
    if [ "${INFLUX_DB_NEEDED}" -eq 1 ] && [ ! -d "${INFLUX_DB_VSM_DATA_DIRECTORY}"/data/engine/wal ]
    then
        createDirectory "${INFLUX_DB_VSM_DATA_DIRECTORY}"
        echo "${INFLUX_DB_TOKEN}" > "${INFLUX_DB_VSM_DATA_DIRECTORY}"/.token

        logPoint "Starting influx to configure it."
        ./influxd-start.sh >> "${LOG_FILE}" 2>&1

        logPoint "The InfluxDB configuration has not been completed yet. Configuring."

        logPoint "Removing the default influx config directory, will be re-created."
        rm -rf ~/.influxdbv2 > /dev/null 2>&1

        logPoint "Waiting for influxdb to start. This may take a few minutes on some systems."

        INFLUX_DB_RETRY_COUNTER=0
        INFLUX_DB_STARTED=0
        while [ ${INFLUX_DB_STARTED} -eq 0 ] && [ ${INFLUX_DB_RETRY_COUNTER} -lt 60 ]
        do
            sleep 5
            "${INFLUX_DB_DIRECTORY}"/influx ping --host "${INFLUX_DB_URL}" > /dev/null 2>&1
            if [ $? -eq 0 ]
            then
                INFLUX_DB_STARTED=1
                logPoint "InfluxDB running"
            fi
            INFLUX_DB_RETRY_COUNTER=$((INFLUX_DB_RETRY_COUNTER+1))
        done

        if [ ${INFLUX_DB_RETRY_COUNTER} -ge 60 ]
        then
            logPoint "${INFLUX_DB_DIRECTORY}/influx ping --host ${INFLUX_DB_URL} was unsuccessful."
            generateWarningErrorMessage 139
        else
            # set up influxdb org and bucket
            sleep 5 # wait 5 extra seconds after influx says it started to avoid EOF error.
            "${INFLUX_DB_DIRECTORY}"/influx setup --bucket "${INFLUX_DB_BUCKET}" --token "${INFLUX_DB_TOKEN}" --org "${INFLUX_DB_ORG}" --username="${INFLUX_DB_USER}" --password="${INFLUX_DB_PASSWORD}" --host="${INFLUX_DB_URL}" --force >>"${LOG_FILE}" 2>&1

            # set retention policy for influxdb to 2 years
            INFLUX_DB_BUCKET_ID=$("${INFLUX_DB_DIRECTORY}"/influx bucket ls --org "${INFLUX_DB_ORG}" --host "${INFLUX_DB_URL}" | grep "${INFLUX_DB_BUCKET}" | cut -f1) > /dev/null 2>>"${LOG_FILE}"
            "${INFLUX_DB_DIRECTORY}"/influx bucket update --host "${INFLUX_DB_URL}" --id "${INFLUX_DB_BUCKET_ID}" --retention 104w >>"${LOG_FILE}" 2>&1
            mkdir -p "${INFLUX_DB_VSM_DATA_DIRECTORY}"/data/engine/wal
        fi
    else
        logPoint "InfluxDB was already configured. Not running any influx commands at this time."
    fi
    
    if [ "${SETENVOVERRIDE_FILE_EXISTS}" -eq 1 ]
    then
        logPoint "Starting FlexDeploy server, may take a few minutes"
        ./StartFlexDeploy.sh >> "${LOG_FILE}" 2>&1
        # check for the startup message in the logs
        SERVER_STARTED=0
        while [ ${SERVER_STARTED} -eq 0 ]
        do
            sleep 5
            grep "Server startup in" "${FLEXDEPLOY_HOME}"/"${APACHE_TOMCAT_FLEXDEPLOY_LOGS}"/flexdeploy.0.log > /dev/null 2>>"${LOG_FILE}"
            if [ $? -eq 0 ]
            then
                SERVER_STARTED=1
            fi
        done
        logPoint "FlexDeploy server running"

        # display the url for users
        HOST_NAME=$(hostname -A | tr -s ' ' | cut -f1 -d' ')
        echo > tmphost.$$
        echo >> tmphost.$$
        if [ "${FLEXDEPLOY_PORT}" -eq 443 ] 
        then
            echo "   URL for FlexDeploy:  https://${HOST_NAME}/flexdeploy " >> tmphost.$$
        elif [ "${FLEXDEPLOY_PORT}" -eq 8443 ] 
        then
            echo "   URL for FlexDeploy:  https://${HOST_NAME}:${FLEXDEPLOY_PORT}/flexdeploy " >> tmphost.$$
        elif [ "${FLEXDEPLOY_PORT}" -eq 80 ] 
        then
            echo "   URL for FlexDeploy:  http://${HOST_NAME}/flexdeploy " >> tmphost.$$
        else
            echo "   URL for FlexDeploy:  http://${HOST_NAME}:${FLEXDEPLOY_PORT}/flexdeploy " >> tmphost.$$
        fi
        echo >> tmphost.$$

        # dump to the screen
        cat tmphost.$$

        # dump to the logs
        cat tmphost.$$ >> "${LOG_FILE}"

        rm -f tmphost.$$
    else
        logPoint "Not starting the server because the previous install didn't have a setenvoverride.sh script and it may need adjustments prior to starting."
        generateWarningErrorMessage 129 ${UPGRADE_BACKUP_TOMCAT_DIRECTORY}/${APACHE_TOMCAT_FLEXDEPLOY_BIN} ${FLEXDEPLOY_HOME}/${APACHE_TOMCAT_FLEXDEPLOY_BIN}
        
    fi

    # go back to where you started
    cdOrFail "${CURRENT_PWD}"
}

# function: stopFlexDeploy
#
# parameters
#    none
#
# return
#    none
#
# function will stop the tomcat server, if running

function stopFlexDeploy
{
    if [ "${EXECUTION_STATE}" -eq "${UPGRADE_FROM_WLS}" ]
    then
        PID=$(ps -aef | grep java | grep wlserver | grep -v grep | tr -s ' ' | cut -f2 -d' ')
        variableNotEmpty "${PID}"
        if [ $? -eq 0 ]
        then
            generateFatalErrorMessage 130 
        fi
    fi

    if [ "${PREREQ_MODE}" -eq 0 ] && [ "${DOCKER_INSTALL}" -eq 0 ]
    then

        CURRENT_WAIT_TIME=0

        PID=$(ps -aef | grep java | grep "${FLEXDEPLOY_HOME}"/"${APACHE_TOMCAT_FLEXDEPLOY_BIN}" | grep -v grep | tr -s ' ' | cut -f2 -d' ')
        variableNotEmpty "${PID}"
        if [ $? -eq 0 ]
        then
            if [ "${EXECUTION_STATE}" -eq "${UPGRADE_FROM_WLS}" ]
            then
                generateFatalErrorMessage 138 "${PID}" "${FLEXDEPLOY_HOME}"/"${APACHE_TOMCAT_FLEXDEPLOY_BIN}"
            fi
            logPoint "Stopping FlexDeploy server"
            # stop the tomcat server
            CURRENT_PWD=$(pwd)
            cdOrFail "${FLEXDEPLOY_HOME}"/"${APACHE_TOMCAT_FLEXDEPLOY_BIN}"

            ./StopFlexDeploy.sh  >> "${LOG_FILE}" 2>>"${LOG_FILE}"

            # go back to where you started
            cdOrFail "${CURRENT_PWD}"

            # check for the startup message in the logs
            SERVER_STOPPED=0
            while [ ${SERVER_STOPPED} -eq 0 ]
            do
                sleep "${STOP_WAIT_TIME}"

                # keep track of loops waiting to for server to shutdown
                CURRENT_WAIT_TIME=$(expr "${CURRENT_WAIT_TIME}" + "${STOP_WAIT_TIME}")

                PID=$(ps -aef | grep java | grep "${FLEXDEPLOY_HOME}" | grep -v grep | tr -s ' ' | cut -f2 -d' ')
                
                if variableEmpty "${PID}"
                then
                    SERVER_STOPPED=1
                elif [ "${CURRENT_WAIT_TIME}" -ge "${MAX_TIME_TO_WAIT_FOR_STOP}" ]
                then
                    # taking too long to shutdown, need to kill
                    kill -9 "${PID}" || generateFatalErrorMessage 147 "${PID}"
                    SERVER_STOPPED=1
                fi
            done

            logPoint "FlexDeploy server stopped"
        fi 

        #Verify that influxd isn't running.
        PID=$(ps -aef | grep "${FLEXDEPLOY_HOME}/influxdb/influxd" | grep -v grep | tr -s ' ' | cut -f2 -d' ')
        
        if ! variableEmpty "${PID}"
        then
            logPoint "Stopping Influx."
            "${FLEXDEPLOY_HOME}"/"${APACHE_TOMCAT_FLEXDEPLOY_BIN}"/influxd-stop.sh  >> "${LOG_FILE}" 2>&1
        fi
        
    fi
}
