#!/bin/bash

# function: logPoint
#
# parameters
#    $1 - log message
#
# return
#    none
#
# function will add a message to the current log with date time stamp

function logPoint
{
    LOG_MESSAGE=${1}

    DTTM=`date +'%x %X'`

    echo "${DTTM} - ${LOG_MESSAGE}" >> ${LOG_FILE}
    echo "${DTTM} - ${LOG_MESSAGE}" 
}

# function: rollLogs
#
# parameters
#    none
#
# return
#    none
#
# function will delete log files older than 2 weeks
function rollLogs
{
    # delete anything older than 60 days
    # LOG_FILE now contains date and time
    find ${LOG_DIRECTORY}/* -type f -mmin 86400 -exec rm -Rf {} \;
}

# function: generateWarningErrorMessage
#
# parameters
#    $1 - error code
#    $2 to $11 - optional data parameters that can be used to for dynamic content in message
#
# return
#    error code
#
# function simply adds on an additional parameter to let the common function know if it is a
# warning messasge so it will allow execution to continue

function generateWarningErrorMessage
{
    generateErrorMessage ${WARN} $@
    WARN_CODES="${WARN_CODES} ${1}"
    WARN_RC=$?

    return ${WARN_RC}
}

# function: generateFatalErrorMessage
#
# parameters
#    $1 - error code
#    $2 to $11 - optional data parameters that can be used to for dynamic content in message
#
# return
#    error code
#
# function simply adds on an additional parameter to let the common function know if it is a
# fatal message so it will stop execution 

function generateFatalErrorMessage
{
    logPoint "Check the installation logs at ${LOG_FILE} for error."
    ERROR_CODES="${ERROR_CODES} ${1}"
    generateErrorMessage ${EXIT} $@
}

function exitFatalErrorMessage
{
    generateFatalErrorMessage $@
    exit ${1}
}

# function: generateErrorMessage
#
# parameters
#    $1 - warn (0) or exit (1) to determine if the execution will continue or not
#    $2 - error code
#    $3 to $12 - optional data parameters that can be used to for dynamic content in message
#
# return
#    error code
#
# function will lookup the error code message and replace up to 10 dynamic content values in the
# error message using the format of {n} ex. This error message {1} has several dynamic variables {2}.
# With this string, there should be 3 parameters passed in to the function.  Based on the first argument,
# the execution will continue on a warn message and terminate on an exit message

function generateErrorMessage
{
    # WARN is a value of 0, exit is a value of 1
    WARN_OR_EXIT=${1}

    ERROR_CODE=${2}
    REPLACE_DATA1=${3}
    REPLACE_DATA2=${4}
    REPLACE_DATA3=${5}
    REPLACE_DATA4=${6}
    REPLACE_DATA5=${7}
    REPLACE_DATA6=${8}
    REPLACE_DATA7=${9}
    REPLACE_DATA8=${10}
    REPLACE_DATA9=${11}
    REPLACE_DATA10=${12}

    # replace any {n} variables in the error message with values that are passed in, up to 10
    # sed allows anything that isn't in the string to be used as a delimiter, so using | so we can 
    # handle directory paths
    echo -e "\n############################### ISSUE ${ERROR_CODE} ##########################################################" > tmp.$$
    cat ${ERROR_MSG_DIR}/${ERROR_CODE} | sed -e "s|{1}|${REPLACE_DATA1}|g" \
                    -e "s|{2}|${REPLACE_DATA2}|g" \
                    -e "s|{3}|${REPLACE_DATA3}|g" \
                    -e "s|{4}|${REPLACE_DATA4}|g" \
                    -e "s|{5}|${REPLACE_DATA5}|g" \
                    -e "s|{6}|${REPLACE_DATA6}|g" \
                    -e "s|{7}|${REPLACE_DATA7}|g" \
                    -e "s|{8}|${REPLACE_DATA8}|g" \
                    -e "s|{9}|${REPLACE_DATA9}|g" \
                    -e "s|{10}|${REPLACE_DATA10}|g" >> tmp.$$

    # check if there is a specific data file, logData.txt, that if present will be appended at the end of the message
    if [ -f "${LOG_DATA_FILE}" ]
    then
        echo >> tmp.$$
        cat "${LOG_DATA_FILE}" >> tmp.$$
        echo >> tmp.$$

        # cleanup the addition log data file
        rm -f "${LOG_DATA_FILE}"
    fi

    echo -e "\n####################################################################################################" >> tmp.$$
    echo >> tmp.$$

    # dump the updated error message to the log file
    if [ "${WARN_OR_EXIT}" -eq "${WARN}" ]
    then
        logPoint "Warning occurred, execution will continue"
    else
        logPoint "Error occurred"
    fi

    cat tmp.$$ >> "${LOG_FILE}"
    echo >> "${LOG_FILE}"

    # dump the same updated error message to the screen
    cat tmp.$$

    # clean up the updated error message
    rm -f tmp.$$

    # stop the execution with the assigned error code number or continue on a warning, continue if PREREQ_MODE as just
    # trying to understand what is wrong
    if [ "${WARN_OR_EXIT}" -eq "${WARN}" ] || [ "${PREREQ_MODE}" -eq 1 ]
    then
        return "${ERROR_CODE}"
    else
        printWarningSummary
        exit "${ERROR_CODE}"
    fi
}


# function: printWarningSummary
#
# parameters
#    none
#
# return
#    none
#
# function will print out the contents of WARN_CODES and ERROR_CODES.

function printWarningSummary
{
    if [ "X${ERROR_CODES}" != "X" ]
    then
        logPoint "SEVERE - Look through the log for more information about the $(echo "${ERROR_CODES}" | wc -w) SEVERE issue code(s) you faced: ${ERROR_CODES} "
    fi
    if [ "X${WARN_CODES}" != "X" ]
    then
        logPoint "INFO/WARN - Look through the log for more information about the $(echo "${WARN_CODES}" | wc -w) INFO/WARN level issue code(s) you faced: ${WARN_CODES}"
    fi
}
